package Lingua::Flags;

use base 'Exporter';
use MIME::Base64;
use warnings;
use strict;

our @EXPORT = qw.as_html_img as_gif.;

=head1 NAME

Lingua::Flags - Provide small flag icons

=cut

our $VERSION = '0.07';

=head1 SYNOPSIS

Provides small flag icons in different formats.

    use Lingua::Flags;

    my $foo = as_html_img("PT");
    print "<p>This is in $foo language</p>";
     
    open GIF, ">image.gif";
    print GIF as_gif("PT");
    close GIF;

=head1 DISCLAIMER

It is impossible to map flags to languages. Languages cross
countries. And countries cross languages. What language should use the
Spanish flag? The Catalan or Castilian or other? What flag should be
used for Afrikaans? South Africa or Namibia?

I tried to choose the more consensual flags, but in some cases I have
doubts. If you have better ideas, please correct me.

Nevertheless, I think this module is useful.

=head1 DESCRIPTION

Module with small flags icons. You can get them in base64 inline HTML
image or as a gif image.

=cut

our %FLAGS = (
              # Portugues
              PT => q{R0lGODlhFAANAPU7AACHNwCINgCINwCJNwSINQaLNQmJNA+PMxGPMyKXLiWQKzqhKDuhKVefIUyiIne6GpQ3F5Q4FpQ4F5lBFaBoDrVmEf4AAP8AAP0FAPsLAP4ZAP4eAPskAPgxAPE8Ff09GcZbGMBlCsp3B8R1FtxuC+pMAOxVAPVOAPlAAPxJAPteAPNZMulwAPtgAPNiI/hkWe9/a4OGFoS6FKKoEp/LEOeBAfGUAPChAL3F4N2zu/aqj+a9vwAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAZjwIBwOIhcjsjkcUg0Kp/MAEAAeUKHgMMixMFYkVhFjFJCmTLfixBgkJFesA+rk14naKMcbtVS1aUIDxU6Oy42KX8AAA4zIB41Nxp/UgQMDSInG2lqWAAFExabnEwDEqKjTadBADs=},
              # Espanhol
              ES => q{R0lGODlhFAANAPUuAP8AAP8vAP8wAP8xAOppDO5qFfhmIuRxU+l5kdSKHN2WLt+YLueFDP2DDf+TD+CDNuiPNP+dPOm+GPW8BvuGWemvbtHCAN3GAN3HOezAEODRAOTZBuzWAebTH+3THfLYEPPcF//BJvnSMu/jG/vuAvjsBf3uAP7uAP/uAPX3APf5Av/4AP78AP/+AP//AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAZYQIBwSCwaj0hiYMlsOp2oKMpUIp2k2Kxr6/IoEhuueMxVTQiQUYrMbokaBkQFxCavIo7HgRJq1cUsGB8FDAsdf1tRJxYcEhkXGlmSTAMCA5cCT05JnJ2cQQA7},
              # Frances
              FR => q{R0lGODlhFAANAPIHAAc9mAxBmmGEvf8AAP9aWv7+/v/+/v///yH5BAAAAAAALAAAAAAUAA0AAAM4GKogdfAYMqpdzEVJbcVB80ST92HiVppgSnZeq73mII/QGqNzDl88HKd2U/1OC5ePGDQyk72hKQEAOw==},
              # Italiano
              IT => q{R0lGODlhFAANAPIHAACFMwCIN1qyff8AAP9aWv7+/v/+/v///yH5BAAAAAAALAAAAAAUAA0AAAM4GKogdfAYMqpdzEVJbcVB80ST92HiVppgSnZeq73mII/QGqNzDl88HKd2U/1OC5ePGDQyk72hKQEAOw==},
              # Esperanto
              EO => q{R0lGODlhFAANAPQAAACXAACYAACZAAGZAS2qLUa1RlK5UnLGcnbIdn/Mf4DMgIrQio/Sj5XUlaTapKXbpabbpq3ercfox97y3uf15/D58PX79fn8+fv9+/3+/f7+/v///wAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAVA4LZpmOgMQqquqVhFJsrOmXQQy/TI86pRBQDCsuuxNhZGw7E5GVei0mUD4T0T2IQiYXiuAOBwwEsum8/otJoVAgA7},
              # Danish
              DA => q{R0lGODlhFAAPAPIHAP8AAP8QEP+YmP+fn//Q0P/W1v/s7P///yH5BAAAAAAALAAAAAAUAA8AAAM0CLpysjDG9qRl7mqqL+/StwkkWRxFqapH674wvAqnMcxqeBABOO09H4YnHAaLDWIRkDxaEgA7},
              # Romanian
              RO => q{R0lGODlhFAANAPIHAAc9nAxBmmGEY/8AAP9aAP7+AP/+AP//ACH5BAAAAAAALAAAAAAUAA0AAAM4GKogdfAYMqpdzEVJbcVB80ST92HiVppgSnZeq73mII/QGqNzDl88HKd2U/1OC5ePGDQyk72hKQEAOw==},
              # Dutch
              NL => q{R0lGODlhFAANAPIFAAtAmQxBmv8AAKm72/+kpP///wAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAMmKLrc/jDKSRW5OGtduv9gKI5kCQ5oqq5r4L5wHAN0bd+3rO98nwAAOw==},
              # English
              EN => q{R0lGODlhFAANAPeSAAc/mgg+mAg/mQk/mQhAmghAmwlAmgpAmQtAmQpAmgtDnAxBmg1Cmg5Cmw1FnRFFnBFFnRFGnRJHnRFInxFJnxNInhVInR1MnhpMoClRnzVTnTpWniFPoCJQoS1cqS9dqjVirDdkrTpnrz5qsUhZnFdcmURdoVlgnUFssUZwtGZdlXhklWZooGaKwmyPxP8AAP8PD/8vL/8wMP8yMdBpev9NTf9OTv9SUv9TUv9TU/9UU/9VU/9UVP9VVO51evdwc4Jmk4RolJJjipRymZR4n6Jxkqhzk75zirJ8l8J2jMN2jMx6jOJ9h+1+g/d9gJOq0pWs05Wt1J2y1p2z15+x1KCy1aa316O52qW42aa42Ka726i52Km52Km52au72qy82q292rK10bS20a/B3rrD3LrH4PCAhfWcn/2dneWYofafoeihqfaipf+npvutrvytrfuvsPy/v/++vMa/08bB1sbE2dPM29XK2dri8N3k8e3Eye7Gy+jP1+nO1u3P1eTU3uXW3+vU2urX3/bFyPTIy/rBwf/CwPzMzP/MzP/Ozv/V1P/X1/zf4Pzh4/zs7f/z8v/29v/9/P/9/QAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAj4AH8MuuIhwAEEEcTMmCEmAgIEADoEaSJHBY02dlpYCPBAIcMHAiqUUPIG0JQCGYT4KAQGhYQwC8NMMIGEDZ8nIBYsSOCAxBE0gqLUWThnyBo9VEQwGLCgS5ctWMoQWiTJUQwZjBBBwuPFyhanPcL20MEjx40aMGDY4HEDxw6xPV7InUu3bl24O3Dc4GEjbY29PN6KdbrFypc8jxI1khHDUSRFfcZk4eJ0wYAGI6rsSUOEzsI6UQKpKaJBwQGdIaD4OWNkAwWYM8JISEEmDpMVHAxI+eMmyQkIAjouFPMgAAYXd+AsYWHIDJALBB4mHO7wQIAPWg45CQgAOw==},
              # Western Frisian
              FY => q{R0lGODlhFAANAPcAAOweE+0nHO0pH+wrIewwJ+4zKe00Ku03Le44L+9BOO9FPO5GPetKQ/BLQvFXT/JaUvFjW850fOhoZPR2bux5de54dO15dvB2cPV9d/aEfgRGmARHmQdJmglJmgxMmwtOng5NnBBPnRJQnhZTnxZVoiVboiFbpCRepShgpyZhqCxjqC1nrDBhpDNnqTJrrjNsrzZsrjdtrjtwsF10qEN2tEt9uGZ3p299qWB+so97m8R3g1CAuleDulWEu12Gu1iHvmGDtmaDtGWEtmKJvG2LunmPul2KwGWOwGmRwWuRwnGWxZeWtZyZt7KSp7mcr6CgvaylvbOkutGNlsqQm9mQltWSmdaTmvaRjPaYk86apMqir8esutKst9evuPGmpe+3t+24ufq1sPy7t4Giy4qpz5iiw5qvz42q0I+t0Z252aq61ay916S82qa92qS+3K7F4LPI4d7Hz+LHzfvEwPLIyPnS0fjV1P7W1P7Z1v7c2svY6s/c69Dd7Nbh7tbi8N7o8/7j4vfq7P7t7OLr9e7y+Pby9P/y8P/19P/49/P2+v/6+f/8+/j7/fr8/f///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAj6AB0J5FODgwYOIsYQEmjHSQwOEDnsEehIThESGjR44MEHEYQFUoaEyGiCDCE8CS5EkWFQwwkzEgJkCMQkRcYOO7YcGCBmkJIRGT3M4GLoCoMpQDJqKJEDjCJHjNa0MMhhBZsKAB4ESgJUw4cffBwJmlAlCIiMIGxUweOoEZwYSlGkoTNggBc3Kg5qcPGGkaM5VIiIQIsjAgZAjvoYOauBxJFCEARUUONCrwo2fhd9WWLToA8dC7Ac6iKEMYgkf+4YcKCFR4eMJcokClOAwhMVLWNAYYBATCI0J9AayUIgQR4/Q0AYFHEjDkU+r1+zaFJHYKIzQCHS0EMxIAA7},
              # Indonesian
              ID => q{R0lGODlhFAANAPECAP8AAP9/f////wAAACH5BAAAAAAALAAAAAAUAA0AAAIXhI+py+0Po5yh2oux2Lz7D4biSJbmiRYAOw==},
              # Malay
              MS => q{R0lGODlhFAAKAPUAAEUvf0cwflk7ckdjfUpie1pscGx3ZGZ2aMtreOlzcLimMN29F5OPSZSPSZ+WQYGEVQ8+owk6pws7pgU3qwA1rQE0rgc6qAI5rBw8pgA0sEQugEUugFo8ghVCnxxFmxhDnR9ImCxQjzlYhj1ahClQkhxAqR1Bq859is+DkNqMldGPnOyFg+yGg+2IhOyIhe2Ni+6Niu6Oi+2OjOuQj+mPkO2Tke+amO+cmvCcmfKurAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAAoAAAZ8QAolIgpdhBQMIsFsNikTQ8ETGkEoJRRsG5vJYDIZBcSwdBwLAlaFu+HetjaO8ml0Ig+FKKnd+mdbGRkDByQdHRQZJScujY0tkCwbkwICABoAGxwpNjc2n6A2M141MjU0NTUzqaytNSywK7KysbO0sCyuurupOb6/wMHBQQA7},
              # Albanian
              SQ => q{R0lGODlhFAANAPZRAAAAAAIAAAgAAAoAAA4AABgAACAAACEAACUAACkAACsAADwAAEEAAEUAAEYAAEsAAFcAAFkAAFsAAFwAAGsAAG8AAHEAAHMAAHYAAHcAAHoAAHwAAH0AAH8AAIIAAIUAAIkAAIwAAJAAAJQAAJUAAJcAAJgAAJwAAJ4AAKAAAKMAAKUAAKYAAKcAALMAALwAAL8AAMIAAMYAAMcAAMgAAMkAAMoAAMsAAMwAAM0AANAAANMAANcAANwAAOAAAOEAAOcAAOkAAOsAAOwAAO0AAO4AAO8AAPEAAPMAAPUAAPcAAPkAAPoAAPsAAPwAAP0AAP4AAP8AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAd6gFGCg4SFhoVORj1RQj5ITYeEODo6FikWLUA5S5GCNBgXIQseIyIvkJ02EAEHBg4DEStInVExEwgAuQIUHUe0MhkVCQAEDxwbRbQ8LDAaBQ0fLyREtEo3JigKEiUuQlC0UTsgMwwgPydB4FFMSUwqNU+o6oJDvvP3UYEAOw==},
              # Turkish
              TR => q{R0lGODlhFAANAPUsAP8AAP8BAf8CAv8EBP8FBf8ICP8JCf8MDP8PD/8QEP8REf8SEv8TE/8UFP8VFf8YGP8aGv8fH/8lJf8mJv8nJ/8rK/8vL/84OP9NTf9bW/9cXP9eXv9oaP94eP96ev97e/99ff+fn/+iov+np/+wsP+ysv+zs/+1tf/MzP/Q0P/l5f/o6P/r6wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAZNQIBwSCwaj8gkYJAYKIsCzEjEoQSewowKFFlABFgF6kQYHiwJpYT1GQYwpYmykSKVhY4L4rlZdR4JDGBYAxomIR4Vg1gABQoGjJGSSUEAOw==},
              # Latin
              LA => q{R0lGODlhFAANAPQeAGgmXmgnXmknXmkoXmwqX20qYG0rYG8sYQk+mAo/mQZBnAdCnAxBmhZNonWTxY9FbpFHb/0AAP8AAI2q1I+s1ZGt1pOv15yx1eDs9+Hs9+Lt9+Tu+P39/v3+/v///wAAACH5BAAAAAAALAAAAAAUAA0AAAVSoCSOZGmeaEpGUSq88BAUz0MEAwwvfK80lAxm0lD0eoykEuHoeDwdB0JJrSYuz+clUe0ymJwnR9o98n6WjaZSNC90rwHgAHkYcHBBiqXq+/8jIQA7},
              # Russian
              RU => q{R0lGODlhFAANAPMIAGEqYwc9mAdCnAxBmmGEvf4AAP8AAP7+/v///wAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAQyEMlJq70463q6/yBIjGRpmkGqriw7vHAsy0Jt3zgO7Hzv+4WgcEgkGo7IpHLJbDqfyggAOw==},
              # Norwegian
              NO => q{R0lGODlhFAAMAPQRABxNoCtZpztorztqsjtrslh+u12Cvl2FwGuNw2uPxf8AAP+AgP+Bgf6Li/CSmPCTme/Izf7GxgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAAwAAAU8oCIqTDI8Y6qm5bm+Y4vC75IQDj02fBMhAkhvSDwYD4YAoHBsOpvJ5fM5/AWJ2EbKhtOtZF4VOBwzzXQhADs=},
              # Afrikaans
              AF => q{R0lGODlhFAAKAPUAAAAAAAwIAA0IAE01AFA3AFc8AAB2SQB3SwB3TQF4TB1+Q+I1JeI4KOQ6K5Z3CaF+COVBMuZCNO11au11a+9+dC+DPnqRJRWCWRqGXB2KYyGKYS+Rai6QayuQbIGRI4uXIgAcnQAgnwMioQ0qpA8rpVBkvVBlvltuwnq3oIq4o4m5pIC4qdannNaonc+2q9mzqdm0qtjEu4Wfx5WszZSwyam/1AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAAoAAAZwwMPhgopRIIskY8lsWioHxCb1kiCbWEbB4YEeOimYJMJomM9mgIDA9XZUrpZ8PgfYBYPHR5FIYDKAgYF2anl7CH4aiouLamxdQhsrNDKVlpZbkAeSMyUkIqChoU+InCYjICGqq6xRRDUnqCGprLO0QQA7},
              # Irish
              GA => q{R0lGODlhFAAKAPIAAP9xAP9zAACTQACVQ/+iVl68if/+/v///yH5BAAAAAAALAAAAAAUAAoAAAMwOKpS95AAEGpYzMEnqcVDs3GTdWHiKJnnkm6dCb5R+aEa7N1uXrMzH2nXyowOHVsCADs=},
              # Swedish
              SV => q{R0lGODlhFAAKAPMAALOvPLSvPL2zN/TNGv/UEvzRFt/EJeDEJQRdlwZelgBbmQBamh9piSNshwAAAAAAACH5BAAAAAAALAAAAAAUAAoAAARCUEmJAglorcn7qleidaSiARdiaizLvG8jEEJjwzhj7PsxEIMDb0jk+YBC4iGZk9EaudxEg0BhWq1SwoIpebchryQCADs=},
              # Breton
              BR => q{iVBORw0KGgoAAAANSUhEUgAAABQAAAAKEAYAAADkxaKlAAAABmJLR0QA/wD/AP+gvaeTAAAACXBIWXMAAABIAAAASABGyWs+AAAACXZwQWcAAAAUAAAACgB3ehUOAAACUUlEQVQ4y8XOvUsbYQCA8ee9a6J2qGDIIIIElUAkeKgEcc0iajWLiItSrHRRRNeDIEgRB0FwcFPoEKEOwToUGziMg1YM4vlFbjCCKSKRkOFAsRd5r0tXpwg+f8DDT5zpZ/qZ7rql3dJuaRc61jrWOtbAmDfmjXkYOB44HjiGVCaVSWWgc6pzqnMKri+uL64vwBGOcET591J2KbuUdb5YI9aINeKNESRIkGoq7J36oD6oD/A88TzxPAE3sZvYTQy8fq/f64er4avhq2EQDaJBNEA+kU/kE/Dke/I9+aAuVBeqC/FhtWu1a7WLX47u6I4uvuLHjx9PxcCgHtSDOgiv8AovZO4z95l76DvpO+k7geR6cj25DoN3g3eDd5CuT9en6yE0GhoNjUJgO7Ad2KZ6rnWuda5VxPPd+e58t/jBOOOMU64UKE4PTw9PD93c7ebt5u0mhGfDs+FZ2D/fP98/h+hWdCu6BWnSpIFIf6Q/0g85O2fnbKgN1AZqA+Wfslf2yt6/H23TNm2z6rPQhCY0qioGjllj1phV/vZoPpqPJjQeNB40HoBVtIpWEcIr4ZXwClwWL4uXRWjJtmRbslCYLkwXphGeNk+bp00mmnqaepp63IK6oW6oG+p3JBJZKQ/Ewt7C3sKe80ksi2WxDDIu4zIOSo1So9SANKQhDVA0RVM0kJrUpAZKu9KutIObdJNuEp874864M7zHxsbmT+W0/8CdyZ3Jncly9LWGr50YSg2lhlKO+daQl3q3WFosLZaE8daQlxLNR81HzUflyFtDXuofJm37+nnT0cEAAAAASUVORK5CYII=},
              # Deutch
              DE => q{R0lGODlhFAAMAPIHAAAAAB4AAOAAAP8AAP4YB/61OP3PQP7OQCH5BAAAAAAALAAAAAAUAAwAAAMrCLrc/jDKFqq9GIvNu/dDKI5kaZ4joa5s2xZwLM+zYd94nh987//AoNCXAAA7},
              # Bosnian
              BS => q{R0lGODlhFAAKAPUAAE1ZaF1lYWJqamNrampxbW91b3J3am91cGdxe3N5ct7CHuLFHOTGHeXHHvzXEv/ZEf/bEODDIAsphwkoigwqiA0riA4siBEvihYzjBc0jRk2jh05kB87kUdepUphpk5kqFRpq1RprVxwr1twsF9zsmZ5tGd6tGl8t26AuHaHvX+PwgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAAoAAAZcwEol4yEhFJCkA7J0BCbCaOWCAhUYzKzDCZVGSycBtsl8eqOYj2qwSDK5Z6Fk00kR2kpzvCLRjEgHbVt6e1MoIYFlXYVCJicDDHCMFxVpKgIKTxSFm0J0KQYAi0EAOw==},
              # Slovene
              SL => q{R0lGODlhFAAKAPQAAFRGd+wcJO0cJABaogRboQFepABepidxpleKuViLuWmRu1qWxJu61py717nR4MbO38TR4vD0+PH0+P7+/////wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAAoAAAU7ICWOZGmKkuNEZ0s9xwG55GIjTJPYfN8PQIJCQQAaj0eDcslsNgHQqHQ6DVgDgqxVcM1yu96weEwuj0MAOw==},
              # Somali
              SO => q{R0lGODlhFAANAPQeAJjY6JnY6JnZ6ZrZ6ZvZ6Z7a6p/a6p/b6qHb6qLc6qPc6qTc66zg7a3g7bPi7rrl787s9M/t9NDt9NHt9dPu9dTu9dTv9dXv9t/z+OX1+eb1+fX7/Pf8/fr9/f///wAAACH5BAAAAAAALAAAAAAUAA0AAAVJoCCOZGmeg5ierEAQLQs4DhCT7zBVw3sHCwhms8FEFoHbgJHxeDSN1Q0g6XAutpsgcKA8HJZE8mdQAACIwjg2CCTbAal2TieFAAA7},
              # Polish
              PL => q{R0lGODlhFAANAPECAP8AAP9/f////wAAACH5BAAAAAAALAAAAAAUAA0AAAIXlI+py+0Po5yh2osx2Lz7D4biSJbmiRYAOw==},
              # Icelandic
              IS => q{R0lGODlhFAAMAPIHAAM6lgc9mAxBml+Bu2GDvf8cHP5VVZmu0yH5BAAAAAAALAAAAAAUAAwAAAM9KBpkRkHJSZmDNEv7ot5BAAzOAIRoGhLscDjHwM4069xFce98n/fAmQsmq9VQo9JJpdo0Op8MBxOdTD3RBAA7},
              # Finish
              FI => q{R0lGODlhFAALAPQaAAY8lwc9mAg9mAk/mQpAmQtAmQxBmg1CmhtMoCFSoyVVpDFeqTNfqkRtsUVtsWGDvWSGvmmJwGmKwJ+016S42ay+3K2+3K2/3evv9vP1+v///wAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAAsAAAVEoCaK1RFEY6qm5bm+o2VK8JjdN2VCeN83wIZjYSAkgkgkYLkUFAwDplTKqFYVRYR1u8V4vRPT40smqy6z2quFUqvY7hAAOw==},
              # Croatian
              HR => q{R0lGODlhFAAKAPU2AHpaewg/mQk/mQlAmQtAmQtAmgxBmhpGlzdaoG5jlK9KYatide0JD/IKDv4BAf8AAP0FBf8KCv8LC/8NDeQZIesiJ/skJNhjbNptdNp2eelOTOROVfV7evd/fv9ycv91dfl+fv9+folZgrl7h/+Afrmds6O00qa62qm726m829qEhN+Eg9SPmv+AgO69ve7c3PjHx//Jyf/KyvLc3PLy8vTz8////wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAAoAAAZbwIdwSCwajRBGhdJwHCfQaESzUABGFkk0Gut6ZS7M5pKBec+2tNr24nw8Ktp6Pp91Qq1VjW5D+f8pJiwkICUnf4gGiosGBAgiCQeMjASVlpUCAwEFl5eTn6CKQQA7},
              # Hungarian
              HU => q{R0lGODlhFAANAPIFAACHNgCIN/8AAKTUuP+kpP///wAAAAAAACH5BAAAAAAALAAAAAAUAA0AAAMmKLrc/jDKSRW5OGtduv9gKI5kCQ5oqq5r4L5wHAN0bd+3rO98nwAAOw==},
              # Welsh
              CY => q{R0lGODlhFAAKAPYAAHw+MDp1NDN5NDJ9NU9WLkldL1VcMltYMmlMMW1PMnRDMH9KOWVSMmFXM2dUMnVSOE9mM6YJJq4OLLYNLbgJKrsJK7oLK7kNLbwOL60TLbMRLbwRL7EbMrUYNb0YNJ4lL5ItL4oyL50oMIU8MZUxMsAML8IOL8oNMsEYN70jQL8pRcYsSMg3UoRTRL5FVr5LYLdSZcRAV8tAWslGXsVJYM5OZs5gdc5neNVqftJsfx6NNg+ZNgucNgycNxSVNhSZNyOGNCOHNSOJNSqTPgWiNwihNwOmOAekOAugOAmjOACqOACrOAGqOQCsOT6DQB6wThu0Thy0T9VugdNzhNeAkdCHldmGldaNm9iKmdyKmdGSmtSWotObpdqdqOSksOW1vue8xObEy+nGzerP1O3P1evQ1e3S1+3W2uvY3PDS19nh1dzl2/Li5fHm6PTl6Pbn6vLr7OD15/v09fn29/z5+f78/P///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAAAAAAALAAAAAAUAAoAAAe2gHaCgm5oYnKCdGNzc3CDg2xhVy8rNG52ZjgsVFNoc2N0g2dcWzAdK2ZeKxYYGFJuVVJ1dYJdVXNYEjJjLBMbJjJgUjRfs3FxWjdrLBEpNCgbJR42WTUuasdQT04LLRwTExWtGBsnHg9OUVBQSu0/IyIdGSAfJSQQDgAKO03t/kg6DDAIAERAiA8BfghRgKSJQ4dKICRIEKRIgQMIBPQg4KBBj4f9lPT4UaSIEiRIiBgRiaTkw0AAOw==},
              # Swahili
              SW => q{R0lGODlhFAAKAPQAAAAAAAUDAAYEAB8aAzEmATEmAzktAjktBIZyDYVxEIV8EIR6HRu1Ox61OiK2OlG+M1i/MqC1H4+1I5SzIZC1IoqtXJavUoSuY4OuZDqvszqws0KxrQal2wCj3QCi4AAAACH5BAAAAAAALAAAAAAUAAoAAAVVYCOOZMM4UIQMQFuS50MpR9AKxSs6Mm0DAcOiUopJfK3g4qLhkHizWlLI5Hg6u8dRCqQ2r50Ob/tTVsHhzgTZXX7T8ILgdlhgMla4vgUYJCwbeXp6IQA7},
);
	

=head1 FUNCTIONS

=head2 as_html_img

Returns an inline HTML image. For now, it just works with
Netscape/Gecko based browsers.

=cut

sub as_html_img {
   my $cc = shift;
   if (exists($FLAGS{$cc})) {
      return "<img alt=\"$cc\" src=\"data:image/gif;base64,$FLAGS{$cc}\"/>";
   } else {
      return undef;
   }
}

=head2 as_gif

Returns the gif code. Normally used as:

  open GIF, ">image.gif";
  print GIF as_gif("PT");
  close GIF;

=cut

sub as_gif {
  my $cc = shift;
  if (exists($FLAGS{$cc})) {
    return decode_base64($FLAGS{$cc});
  } else {
    return undef;
  }
}

=head1 AUTHOR

Alberto Simoes, C<< <ambs@cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to
C<bug-lingua-flags@rt.cpan.org>, or through the web interface at
L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Lingua-Flags>.
I will be notified, and then you'll automatically be notified of progress on
your bug as I make changes.

=head1 COPYRIGHT & LICENSE

Copyright 2007-2009 Alberto Simoes, all rights reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1; # End of Lingua::Flags
