use strict;
package Dist::Zilla::PluginBundle::Author::MAXHQ;
# ABSTRACT: MAXHQ's default Dist::Zilla configuration
$Dist::Zilla::PluginBundle::Author::MAXHQ::VERSION = '1.002001';
#pod =encoding UTF-8
#pod  
#pod =head1 SYNOPSIS
#pod  
#pod Put following into your C<My-Module/dist.ini>:
#pod
#pod 	[@Author::MAXHQ]
#pod 	GatherDir.exclude_match = ^[^\/\.]+\.txt$
#pod 	PodWeaver.replacer = replace_with_nothing
#pod 	ReadmeAnyFromPod = no
#pod
#pod =head2 DESCRIPTION
#pod
#pod The bundles' behaviour can be altered by the following options:
#pod
#pod =for :list
#pod * C<GatherDir.exclude_match> - a regex specifying which files or directories to 
#pod   ignore (they are not processed and thus not added to the distribution tarball).
#pod   This option can be specified several times for different regexes.
#pod * C<PodWeaver.replacer> - Which replacer to use for POD sections.
#pod   See L<Pod::Elemental::PerlMunger>.
#pod   Currently possible values: "replace_with_nothing",
#pod   "replace_with_comment" (default), "replace_with_blank"
#pod * C<nodoc> - Set this to 1 to prevent the generation of HTML documents under doc/
#pod
#pod =head1 OVERVIEW
#pod
#pod Currently this plugin bundle is equivalent to:
#pod 	
#pod 	#
#pod 	# Files included
#pod 	#
#pod 	[GatherDir]
#pod 	exclude_match = \A[^\/]+\.ini\Z
#pod 	exclude_match = \A[^\/]+\.tar\.gz\Z
#pod 	prune_directory = ^\.build$
#pod 	prune_directory = ^\.git$
#pod 	prune_directory = ^\.svn$
#pod 	prune_directory = ^extlib$
#pod 	prune_directory = ^CVS$
#pod 	include_dotfiles = 1
#pod 	
#pod 	[PruneCruft]
#pod 	[ExecDir]
#pod 	dir = bin
#pod 	
#pod 	[ShareDir]
#pod 	dir = share/dist/My-Module
#pod 	
#pod 	#
#pod 	# Conversion and replacements
#pod 	#
#pod 	[PkgVersion]
#pod 	die_on_existing_version = 1
#pod 	die_on_line_insertion   = 1
#pod 	
#pod 	[NextRelease]
#pod 	format = '%-9v %{yyyy-MM-dd}d'
#pod 	
#pod 	[PreviousVersion::Changelog]
#pod 	[NextVersion::Semantic]
#pod 	major = *NEW FEATURES, *API CHANGES
#pod 	minor = +ENHANCEMENTS
#pod 	revision = REVISION, BUG FIXES, DOCUMENTATION
#pod 	numify_version = 1
#pod 	format = %d.%03d%03d
#pod
#pod 	[PodWeaver]
#pod 	config_plugin = @Author::MAXHQ
#pod 	replacer      = replace_with_comment
#pod
#pod 	#
#pod 	# Prerequisites
#pod 	#
#pod 	[AutoPrereqs]
#pod 	[Prereqs::AuthorDeps]
#pod 	[PrereqsClean]
#pod
#pod 	[Prereqs / MAXHQ]
#pod 	-phase = devlop
#pod 	-relationship = requires
#pod 	Pod::Elemental::Transformer::List = 0
#pod 	Pod::Coverage::TrustPod = 0
#pod
#pod 	#
#pod 	# Auto generation --- distribution files
#pod 	#
#pod 	[ModuleBuild]
#pod 	[MetaYAML]
#pod 	[Manifest]
#pod 	[License]
#pod 	[ReadmeAnyFromPod]
#pod
#pod 	#
#pod 	# Auto generation --- tests
#pod 	#
#pod 	[Test::Inline]
#pod 	[ExtraTests]
#pod 	[Test::Perl::Critic]
#pod 	[PodCoverageTests]
#pod 	[PodSyntaxTests]
#pod
#pod 	#
#pod 	# Auto generation --- docs
#pod 	#
#pod 	[Pod2Html]
#pod 	dir = doc
#pod
#pod 	# Release
#pod 	[TestRelease]
#pod 	[ConfirmRelease]
#pod
#pod =cut 

use Moose;

# choose the easy way of configuring a plugin bundle
with 'Dist::Zilla::Role::PluginBundle::Easy';

# add all plugins configured below to the prerequisites of this pluginbundle module
# (requires setting "bundledeps_phase = runtime" in this module's dist.ini)
with 'Dist::Zilla::Role::BundleDeps';



#pod =for Pod::Coverage mvp_multivalue_args
#pod
#pod =cut
#
#"If you want a configuration option that takes more than one value, you'll need
#to mark it as multivalue arg by having its name returned by
#C<mvp_multivalue_args>."
#
#Queried by L<Dist::Zilla>.
#
sub mvp_multivalue_args { return qw(GatherDir.exclude_match) }

#pod =method add_plugins_if_wanted
#pod
#pod Adds the given plugins unless there is a configuration option given to the
#pod plugin bundle that tells not to use it.
#pod
#pod E.g. plugin 'PruneCruft' is used unless the following is given:
#pod
#pod 	[@Author::MAXHQ]
#pod 	PruneCruft = no
#pod
#pod =cut
sub add_plugins_if_wanted {
	my ($self, @plugin_specs) = @_;
 
	my @plugin_specs_filtered = ();
	foreach my $this_spec (@plugin_specs) {
		my $moniker = ref $this_spec ? $this_spec->[0] : $this_spec;
 		# skip plugin if told by user
		next if grep { $moniker =~ /^\Q$_\E$/ && $self->payload->{$_} =~ /^no$/i } keys %{$self->payload};
		# otherwise add to list
		push @plugin_specs_filtered, $this_spec;
	}
 	$self->add_plugins(@plugin_specs_filtered);
}

#pod =for Pod::Coverage configure
#pod
#pod =cut
#
#Required by role L<Dist::Zilla::Role::PluginBundle::Easy>.
#
#Configures the plugins of this bundle.
#
sub configure {
    my $self = shift;
	
	# build this array by merging...
	my @exclude_matches = (
		# ...the parameter (or an empty ref)
		@{ $self->payload->{'GatherDir.exclude_match'} || [] },
		# ...with the default options (exclude certain files at top level)
		qw(
			\A[^\/]+\.ini\Z
			\A[^\/]+\.tar\.gz\Z
		)
	);
 
    $self->add_plugins_if_wanted(
		#
		# Files included
		#
		['GatherDir' => {                 # skip files on top level
			exclude_match => \@exclude_matches,
			prune_directory => ['\Aextlib\Z', '\A\.build\Z', '\A\.git\Z', '\A\.svn\Z', '\ACVS\Z'],
			include_dotfiles => 1, # needed e.g. for ".devdir"
		}],
		'PruneCruft',                     # prune stuff you probably don't want

		['ExecDir' => {                   # install contents of bin/ as executable
			dir => 'bin',
		}],		
		'ShareDir::ProjectDistDir',       # include all files in /share/dist/My-Dist
		
		#
		# Conversion and replacements
		#
		['PkgVersion' => {                # insert version number in first blank line
			die_on_existing_version => 1,
			die_on_line_insertion   => 1,
		}],
		['NextRelease' => {               # replace {{$NEXT}} in "Changes" file with new version and
			format => '%-9v %{yyyy-MM-dd}d', # date (MUST be included before NextVersion::Semantic)
		}],
		'PreviousVersion::Changelog',     # fetch previous version from changelog
		                                  # alternatively run: V=0.00100 dzil release
		['NextVersion::Semantic' => {     # generate next version based on type of changes
			major => '*NEW FEATURES, *API CHANGES',
			minor => '+ENHANCEMENTS',
			revision => 'REVISION, BUG FIXES, DOCUMENTATION',
			numify_version => 1,          # use x.yyyzzz convention instead of x.y.z
			format => '%d.%03d%03d',
		}],
		# Please note that * and ! are mainly there to enforce correct ordering
		# as CPAN::Changes::Release (used in NextVersion::Semantic) just sorts
		# groups alphabetically
		
		# weave your Pod together from configuration and Dist::Zilla
		# (turns "# ABSTRACT" into POD, processes =method and short lists etc.)
		# To exclude files from PodWeaver see: http://blogs.perl.org/users/polettix/2011/11/distzilla-podweaver-and-bin.html
		['PodWeaver' => {
			config_plugin => '@Author::MAXHQ',
			replacer      =>
				$self->payload->{'PodWeaver.replacer'}
				|| 'replace_with_comment', # replace original POD with comments to preserve line numbering 
				
		}],

		#
		# Prerequisites
		#
		'AutoPrereqs',                    # extract prereqs from your modules
		'Prereqs::AuthorDeps',            # "adds Dist::Zilla and the result of "dzil
		                                  # authordeps" to the 'develop' phase prerequisite list"
		'PrereqsClean',                   # "Cleans up mess from other Prereq modules"
		['Prereqs' => 'MAXHQ-DEV', {      # special prerequisites (name MAXHQ-DEV 
			'-phase' => 'develop',        # disambiguates the Prereqs plugin in
			'-relationship' => 'requires',# case it's also specified in users' dist.ini)
			'Pod::Elemental::Transformer::List' => 0, # not automatically installed with Dist::Zilla
			'Pod::Coverage::TrustPod' => 0,           # not automatically installed with Dist::Zilla
		}],
		
		
		#
		# Auto generation --- distribution files
		#
		'ModuleBuild',                    # build a Build.PL that uses Module::Build to install the distribution
		'MetaYAML',                       # generate META.yml
		'Manifest',                       # generate Manifest (list of all files)
		'License',                        # generate LICENSE
		'ReadmeAnyFromPod',               # generate README (with dist's name, version, abstract, license)

		#
		# Auto generation --- tests
		#
		'Test::Inline',                   # generate .t files from inline test code (POD)
		
		# auto-generate various tests
		'ExtraTests',
		'Test::Perl::Critic',
		'PodCoverageTests',
		'PodSyntaxTests',

		#
		# Auto generation --- docs
		#
		['Pod2Html' => {                  # generate HTML documentation
			'dir' => 'doc',
		}],

		#
		# Release
		#
		'TestRelease',                    # test before releasing
		'ConfirmRelease',                 # ask for confirmation before releasing
    );
}
 
no Moose;
__PACKAGE__->meta->make_immutable;

{
	package Dist::Zilla::Plugin::ShareDir::ProjectDistDir;
	# ABSTRACT: install a directory's contents as "ShareDir" content
$Dist::Zilla::Plugin::ShareDir::ProjectDistDir::VERSION = '1.002001';
use Moose;
	use namespace::autoclean;

	with 'Dist::Zilla::Role::ShareDir';

 	use File::Spec;
 
	has dir => (
		is	 => 'ro',
		isa	=> 'Str',
		lazy_build => 1,
	);
	
	sub _build_dir {
		my ($self) = @_;
		return File::Spec->catfile('share', 'dist', $self->zilla->name),
	}
	 
	sub find_files {
		my ($self) = @_;
		my $dir = $self->dir;
		return [
			grep { index($_->name, "$dir/") == 0 }
				@{ $self->zilla->files }
		];
	}
	 
	sub share_dir_map {
		my ($self) = @_;
		my $files = $self->find_files;
		return unless @$files;
		return { dist => $self->dir };
	}
	 
	__PACKAGE__->meta->make_immutable;
}

__END__

=pod

=head1 NAME

Dist::Zilla::PluginBundle::Author::MAXHQ - MAXHQ's default Dist::Zilla configuration

=head1 VERSION

version 1.002001

=head1 SYNOPSIS

Put following into your C<My-Module/dist.ini>:

	[@Author::MAXHQ]
	GatherDir.exclude_match = ^[^\/\.]+\.txt$
	PodWeaver.replacer = replace_with_nothing
	ReadmeAnyFromPod = no

=head2 DESCRIPTION

The bundles' behaviour can be altered by the following options:

=over 4

=item *

C<GatherDir.exclude_match> - a regex specifying which files or directories to  ignore (they are not processed and thus not added to the distribution tarball). This option can be specified several times for different regexes.

=item *

C<PodWeaver.replacer> - Which replacer to use for POD sections. See L<Pod::Elemental::PerlMunger>. Currently possible values: "replace_with_nothing", "replace_with_comment" (default), "replace_with_blank"

=item *

C<nodoc> - Set this to 1 to prevent the generation of HTML documents under doc/

=back

=head1 OVERVIEW

Currently this plugin bundle is equivalent to:

	#
	# Files included
	#
	[GatherDir]
	exclude_match = \A[^\/]+\.ini\Z
	exclude_match = \A[^\/]+\.tar\.gz\Z
	prune_directory = ^\.build$
	prune_directory = ^\.git$
	prune_directory = ^\.svn$
	prune_directory = ^extlib$
	prune_directory = ^CVS$
	include_dotfiles = 1
	
	[PruneCruft]
	[ExecDir]
	dir = bin
	
	[ShareDir]
	dir = share/dist/My-Module
	
	#
	# Conversion and replacements
	#
	[PkgVersion]
	die_on_existing_version = 1
	die_on_line_insertion   = 1
	
	[NextRelease]
	format = '%-9v %{yyyy-MM-dd}d'
	
	[PreviousVersion::Changelog]
	[NextVersion::Semantic]
	major = *NEW FEATURES, *API CHANGES
	minor = +ENHANCEMENTS
	revision = REVISION, BUG FIXES, DOCUMENTATION
	numify_version = 1
	format = %d.%03d%03d

	[PodWeaver]
	config_plugin = @Author::MAXHQ
	replacer      = replace_with_comment

	#
	# Prerequisites
	#
	[AutoPrereqs]
	[Prereqs::AuthorDeps]
	[PrereqsClean]

	[Prereqs / MAXHQ]
	-phase = devlop
	-relationship = requires
	Pod::Elemental::Transformer::List = 0
	Pod::Coverage::TrustPod = 0

	#
	# Auto generation --- distribution files
	#
	[ModuleBuild]
	[MetaYAML]
	[Manifest]
	[License]
	[ReadmeAnyFromPod]

	#
	# Auto generation --- tests
	#
	[Test::Inline]
	[ExtraTests]
	[Test::Perl::Critic]
	[PodCoverageTests]
	[PodSyntaxTests]

	#
	# Auto generation --- docs
	#
	[Pod2Html]
	dir = doc

	# Release
	[TestRelease]
	[ConfirmRelease]

=head1 METHODS

=head2 add_plugins_if_wanted

Adds the given plugins unless there is a configuration option given to the
plugin bundle that tells not to use it.

E.g. plugin 'PruneCruft' is used unless the following is given:

	[@Author::MAXHQ]
	PruneCruft = no

=encoding UTF-8

=for Pod::Coverage mvp_multivalue_args

=for Pod::Coverage configure

=head1 AUTHOR

Jens Berthold <jens.berthold@jebecs.de>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Jens Berthold.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
